/* Copyright (c) 2009, NICTA
 * All rights reserved. 
 * 
 * The contents of this file are subject to the Mozilla Public License 
 * Version 1.1 (the "License"); you may not use this file except in 
 * compliance with the License. You may obtain a copy of the License at 
 * http://www.mozilla.org/MPL/ 
 * 
 * Software distributed under the License is distributed on an "AS IS" 
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the 
 * License for the specific language governing rights and limitations 
 * under the License. 
 * 
 * Authors: Choon Hui Teo (ChoonHui.Teo@anu.edu.au)
 *
 * Created: (02/11/2007) 
 *
 * Last Updated: (12/01/2009)   
 */

#ifndef _HINGELOSS_HPP_
#define _HINGELOSS_HPP_

#include "sml.hpp"
#include "binaryclassificationloss.hpp"
#include "model.hpp"
#include <boost/numeric/ublas/vector.hpp>


/**  
 * Class to represent binary hinge loss function:
 * loss = max(0, margin - y*f(x))
 * where f(x) := <w, x> and margin \geq 0. By default margin = 1.
 */
class CHingeLoss : public CBinaryClassificationLoss 
{
   protected:
      void Loss(double& loss, TheMatrix& f);
      void LossAndGrad(double& loss, TheMatrix& f, TheMatrix& l);
      TheMatrix gradConst, biasConst;
      double varScaleFactor, varScaleFactor03;
      double * aux_buffer;
      
   public:    
      CHingeLoss(CModel* &model, CVecData* &data)
          : CBinaryClassificationLoss(model, data), 
          gradConst(data->size(), 1, SML::DENSE),
          biasConst(data->size(), 1, SML::DENSE)
      {
          aux_buffer = 0;
          for (int i = 0; i < _data->size(); i ++)
            biasConst.Set(i, 1.0);
      }
      virtual void setVarScaleFactor(double _varScaleFactor)
      {
          varScaleFactor = _varScaleFactor;
          for (int i = 0; i < _data->size(); i ++)
              gradConst.Set(i, - varScaleFactor / _data->size());
      }

      virtual void setVarScaleFactor03(double _varScaleFactor03)
      {
          varScaleFactor03 = _varScaleFactor03;
          for (int i = 0; i < _data->size(); i ++)
              gradConst.Set(i, 1.0 / varScaleFactor03 / _data->size());
      }
      virtual ~CHingeLoss() {}
      virtual void ComputeDualLossAndGradient(double& loss, TheMatrix& grad, TheMatrix& curVar);
      virtual void ComputeDualLossAndGradient03(double& loss, TheMatrix& grad, TheMatrix& curVar, double lambda);
      virtual void getLabels(boost::numeric::ublas::vector<double> &value);
      virtual void getLabels(TheMatrix &value);
      virtual double ComputePrimalObj(TheMatrix &dualVar, double lambda, TheMatrix &weight, 
                                    int num_pos, bool useBias, double &bias_value);
      virtual void map_alpha_to_w(TheMatrix &alpha, TheMatrix &w, double scale);
      virtual void aux_map_w_to_alpha(TheMatrix &w, TheMatrix &output);
      double find_bias(TheMatrix &disc, const TheMatrix &label, int num_example, int num_pos);
};

#endif
